<?php

namespace Codebase\PageBuilder\Concerns;

use Codebase\PageBuilder\Models\EditablePage;
use Codebase\PageBuilder\Models\StaticPage;
use DOMDocument;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

trait InteractWithEditablePage
{
    public $placeholders = [];

    public static function getPagePath(EditablePage $page)
    {
        $explode = explode('\\', $page->editable_type);
        $path = Str::kebab(end($explode)).DIRECTORY_SEPARATOR;
        $path .= $page->getPageKey();

        return resource_path('static-pages'.DIRECTORY_SEPARATOR.$path.DIRECTORY_SEPARATOR.$page->lang);
    }

    public function getHtmlPath()
    {
        return static::getPagePath($this).DIRECTORY_SEPARATOR."index.html";
    }

    public function getCssPath()
    {
        return static::getPagePath($this).DIRECTORY_SEPARATOR."styles.css";
    }

    protected function getModelClass($slugify = false): string
    {
        return $slugify ? str_replace('\\', '-', static::class) : static::class;
    }

    public function getModelBaseClass()
    {
        $explode = explode('\\', $this->getModelClass()) ?? ['Item'];

        return end($explode);
    }

    protected function getKeyValue()
    {
        return $this->{$this->getKeyName()};
    }

    public function getEditorPageTitleAttribute(): string
    {
        $title = $this->name ?: $this->title ?: $this->slug ?: '' ?: $this->getPageName();

        return "{$title} ".$this->getModelBaseClass();
    }

    public function setGjsDataAttribute($value)
    {
        $this->attributes['gjs_data'] = json_encode($value);
    }

    public function getGjsDataAttribute($value): array
    {
        return json_decode($value, true) ?? [];
    }

    protected function getPlaceholderAttributes($placeholder)
    {
        $attributes = ['item' => $this];
        try {
            $placeholder = html_entity_decode($placeholder);
            $dom = new DOMDocument;
            libxml_use_internal_errors(true);
            $dom->loadHTML("<$placeholder />");
            libxml_use_internal_errors(false);

            $body = $dom->documentElement->firstChild;
            $placeholder = $body->childNodes[0];
            $length = $placeholder->attributes->length;

            for ($i = 0; $i < $length; ++$i) {
                $name = $placeholder->attributes->item($i)->name;
                $value = $placeholder->getAttribute($name);

                if (empty($value) || $value == "''") {
                    $value = true;
                }

                $attributes[$name] = $value;
            }
        } catch (\Throwable $th) {
            //throw $th;
        }

        return $attributes;
    }

    protected function findAndSetPlaceholders($html)
    {
        $re = '/\[\[[A-Z][a-z]*(-[A-Z][a-z]*)*([\s]+[a-z]+(=.+)?)*\]\]/';

        preg_match_all($re, $html, $placeholders);

        $placeholders = $placeholders[0] ?? [];

        foreach ($placeholders as $_placeholder) {
            if (empty($this->placeholders[$_placeholder])) {
                $placeholder = str_replace(['[[', ']]'], '', $_placeholder);
                $attributes = $this->getPlaceholderAttributes($placeholder);

                $view = preg_split('/[\s]+/', $placeholder);
                $view = array_shift($view);
                $view = strtolower($view);

                if (view()->exists("grapesjs::placeholders.{$view}")) {
                    $this->setPlaceholder($_placeholder,
                        view("grapesjs::placeholders.{$view}", $attributes)->render());
                }
            }
        }

        $placeholders = $this->getPlaceholders();
        $html = str_replace(array_keys($placeholders), array_values($placeholders), $html);

        return $html;
    }

    public function getHtmlAttribute(): string
    {
        $html = $this->components;

        if (empty($html)) {
            return '';
        }

        return $this->findAndSetPlaceholders($html);
    }

    public function getCssAttribute(): string
    {
        return $this->styles;
    }

    public function getComponentsAttribute(): string
    {
        $htmlPath = $this->getHtmlPath();
        if (! File::exists($htmlPath)) {
            return "";
        }

        return File::get($htmlPath);
    }

    public function getStylesAttribute(): string
    {
        $cssPath = $this->getCssPath();
        if (! File::exists($cssPath)) {
            return "";
        }

        return File::get($cssPath);
    }

    public function getStyleSheetLinksAttribute(): array
    {
        return [];
    }

    public function getScriptLinksAttribute(): array
    {
        return [];
    }

    public function getAssetsAttribute(): array
    {
        return [];
    }

    public function getStoreUrlAttribute(): string
    {
        return route('dashboard.core.static-pages.editable-pages.store', [$this->id]);
    }

    public function getTemplatesUrlAttribute(): ?string
    {
        return '';
    }

    public function getPlaceholders()
    {
        return $this->placeholders;
    }

    public function setPlaceholder($placeholder, $content)
    {
        $this->placeholders[$placeholder] = $content;

        return $this;
    }

    public function getPageName()
    {
        return match ($this->editable::class) {
            StaticPage::class => $this->editable->name,
        };
    }

    public function getPageKey()
    {
        return match ($this->editable::class) {
            StaticPage::class => $this->editable->key,
        };
    }
}
