<?php

namespace App\Http\Controllers\Dashboard\User;

use App\Domain\Core\Models\Administration\User;
use App\Domain\Core\Models\Branch;
use App\Domain\Core\Models\City;
use App\Domain\User\Datatables\AdminDatatable;
use App\Http\Controllers\Dashboard\DashboardController;
use App\Http\Requests\Dashboard\Core\Administration\UserRequest;
use App\Support\Dashboard\Crud\WithDatatable;
use App\Support\Dashboard\Crud\WithDestroy;
use App\Domain\Core\Enums\CorePermissions;
use App\Support\Dashboard\Crud\WithForm;
use App\Support\Dashboard\Crud\WithStore;
use App\Support\Dashboard\Crud\WithUpdate;
use HsmFawaz\UI\Services\RolesAndPermissions\RolesEnum;
use Illuminate\Database\Eloquent\Model;
use App\Domain\User\Datatables\DriverDatatable;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Arr;
use Illuminate\Validation\Rules\Password;
use Spatie\Permission\Models\Role;

class AdminController extends DashboardController
{
    use WithDatatable, WithForm, WithStore, WithUpdate, WithDestroy;

    protected string $name = 'Admin';
    protected string $path = 'dashboard.user.admins';
    protected string $datatable = AdminDatatable::class;
    protected string $model = User::class;
    protected array $permissions = [CorePermissions::class, 'admins'];

    protected function rules()
    {

        $validation = [
            'name' => 'required|string|max:100',
            'phone' => 'required|string|max:15|unique:users,phone',
            'email' => 'required|email|max:255|unique:users,email',
            'password' => ['required', 'confirmed', Password::min(6)],
            'roles' => 'required|array',
            'avatar' => 'sometimes|image|max:2000',
        ];
        if (request()->isMethod('PUT')) {
            $validation['password'] = ['nullable', 'confirmed', Password::min(6)];
            $validation['phone'] = 'required|max:255|unique:users,phone,' . request()->route('admin');
            $validation['email'] = 'required|max:255|unique:users,email,' . request()->route('admin');
        }

        return $validation;
    }

    protected function formData(?Model $model = null): array
    {
        return [
            'selected' => $model?->getRoleNames(),
            'roles' => toMap(Role::where('guard_name', 'web')
                ->whereNotIn('name', RolesEnum::toValues())
                ->get(['id', 'name']), 'name'),
        ];
    }

    protected function storeAction(array $validated)
    {
        $roles = Arr::pull($validated, 'roles', []);
        $avatar = Arr::pull($validated, 'avatar');
        $model = User::create($validated);
        $model->syncRoles($roles);

        $avatar && $model->addMedia($avatar)->toMediaCollection();
    }

    protected function updateAction(array $validated, Model $model)
    {
        $roles = Arr::pull($validated, 'roles', []);

        $model->update($validated);
        $model->syncRoles($roles);

        $avatar = Arr::pull($validated, 'avatar');
        if ($avatar instanceof UploadedFile) {
            $model->clearMediaCollection();
            $model->addMedia($avatar)->toMediaCollection();
        }
    }
}
