<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;

class UpdateMetaTitle extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'update:meta-title-dynamic {lang? : The subfolder under "views/frontend" (e.g., ar, en)} 
    {--dry-run : Preview changes without modifying files}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Update <title> and <meta> tags in resource files dynamically with GPT-generated text';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $lang = $this->argument('lang')??"**"; // Get the folder name (e.g., ar, en)

        $directory = resource_path('views');
        $files = glob($directory . "/frontend/**/*.blade.php");

        if (!is_dir($directory)) {
            $this->error("Directory not found: {$directory}");
            return 1;
        }

        $isDryRun = $this->option('dry-run');
        $updatedCount = 0;
        $skippedCount = 0;

        foreach ($files as $file) {
            $content = file_get_contents($file);

            if ($content === false) {
                $this->error("Failed to read file: {$file}");
                $skippedCount++;
                continue;
            }

            $originalContent = $content;

            // Process <title> tag
            $content = $this->processTitleTag($content, $file);

            // Process <meta name="description"> tag
            $content = $this->processMetaDescription($content, $file);
            // Process  content with class change-by-command tag
            $content = $this->processChangeByCommandContent($content, $file);

            // Save file only if changes were made
            if ($content !== $originalContent) {
                if ($isDryRun) {
                    $this->info("Dry run - Changes detected for: {$file}");
                } else {
                    file_put_contents($file, $content);
                    $this->info("\033[32mUpdated: {$file}\033[0m");
                }
                $updatedCount++;
            } else {
                $this->warn("\033[33mNo changes made to: {$file}\033[0m");
                $skippedCount++;
            }
        }

        $this->info("Processing complete. {$updatedCount} files updated, {$skippedCount} skipped.");
        return 0;
    }

    /**
     * Process and update the <title> tag.
     *
     * @param string $content
     * @param string $file
     * @return string
     */

    private function processTitleTag(string $content, string $file): string
    {
        // Regex to match <title> tag and capture the content
        $pattern = '/<title>\s*\{\{\s*(.*?)\s*\}\}\s*<\/title>/i';

        if (preg_match($pattern, $content, $matches)) {
            $oldTitle = $matches[1]; // Captures the Blade expression inside {{ ... }}
            $this->info("Old <title> found in {$file}: '{$oldTitle}'");

            // Generate a new title with OpenAI
            $openAIService = new \App\Services\OpenAITranslationService();
            $newTitle = $openAIService->generateTitle($oldTitle);

            if (!$newTitle || empty(trim($newTitle))) {
                $this->error("Failed to generate a valid new title for {$file}");
                return $content;
            }

            $this->info("Generated new <title>: {$newTitle}");

            // Strip any existing quotes to avoid duplication
            $finalTitle = trim($newTitle, '"');
            // Clean and safely format the new title
            $sanitizedTitle = htmlspecialchars($newTitle, ENT_QUOTES, 'UTF-8');
            // Add spaces before and after placeholders

            $finalTitle = preg_replace(
                '/(?<!\s)(\$website_title|\$website_name)(?!\s)/',
                '$1 ',
                $finalTitle
            );
            // Replace the old title with the new one, ensuring proper Blade syntax
            return preg_replace(
                $pattern,
                '<title>{{ "' . $finalTitle . '" }}</title>',
                $content
            );
        }

        $this->warn("No <title> tag found in {$file}");
        return $content; // Return the original content if no match
    }

    private function processChangeByCommandContent(string $content, string $file): string
    {
        // Regex to match elements with class="change-by-command" and capture their inner content
        $pattern = '/<(\w+)\s+class=["\']change-by-command["\'].*?>(.*?)<\/\1>/is';

        if (preg_match_all($pattern, $content, $matches, PREG_SET_ORDER)) {
            $openAIService = new \App\Services\OpenAITranslationService();

            foreach ($matches as $match) {
                $tag = $match[1];         // The HTML tag (e.g., p, h2)
                $oldContent = $match[2]; // The inner content of the tag

                $this->info("Found <{$tag}> with class 'change-by-command' in {$file}: '{$oldContent}'");

                // Generate new content with OpenAI
                $newContent = $openAIService->generateForContent($oldContent,false);

                if (!$newContent || empty(trim($newContent))) {
                    $this->error("Failed to generate new content for <{$tag}> in {$file}");
                    continue; // Skip this element
                }

                $this->info("Generated new content for <{$tag}>: '{$newContent}'");


                // Remove Arabic commas (،) from the new description
                $finalDescription = str_replace('،', '', $newContent);

                // Add spaces before and after placeholders
                $finalDescription = preg_replace(
                    '/(?<!\s)(\$website_title|\$website_name)(?!\s)/',
                    '$1 ',
                    $finalDescription
                );
                // Trim excess quotes and whitespace
                $finalDescription = trim($finalDescription, '" ');

                // Replace the old content with the new one
                $replacement = "<{$tag} class=\"change-by-command\">{{ \"" . $finalDescription . "\" }}</{$tag}>";
                $content = str_replace($match[0], $replacement, $content);
            }
        } else {
            $this->warn("No elements with class 'change-by-command' found in {$file}");
        }

        return $content; // Return the modified content
    }


    /**
     * Process and update the <meta name="description"> tag.
     *
     * @param string $content
     * @param string $file
     * @return string
     */
    private function processMetaDescription(string $content, string $file): string
    {
        // Regex to match <meta name="description"> and capture the content
        $pattern = '/<meta\s+name="description"\s+content="(.*?)"\s*\/?>/i';

        if (preg_match($pattern, $content, $matches)) {
            $oldDescription = $matches[1]; // Captures the old description text

            $this->info("Old <meta description> found in {$file}: '{$oldDescription}'");

            // Generate a new description with OpenAI
            $openAIService = new \App\Services\OpenAITranslationService();
            $newDescription = $openAIService->generateTitle($oldDescription,false);

            if (!$newDescription || empty(trim($newDescription))) {
                $this->error("Failed to generate a valid new meta description for {$file}");
                return $content;
            }

            $this->info("Generated new <meta description>: {$newDescription}");
            // Remove Arabic commas (،) from the new description
            $finalDescription = str_replace('،', '', $newDescription);

            // Add spaces before and after placeholders
            $finalDescription = preg_replace(
                '/(?<!\s)(\$website_title|\$website_name)(?!\s)/',
                '$1 ',
                $finalDescription
            );
            // Trim excess quotes and whitespace
            $finalDescription = trim($finalDescription, '" ');

            // Replace the old description with the new one, ensuring proper Blade syntax
            return preg_replace(
                $pattern,
                '<meta name="description" content="{{ "' .$finalDescription . '" }}" />',
                $content
            );
        }

        $this->warn("No <meta description> tag found in {$file}");
        return $content; // Return the original content if no match
    }

}
