<?php

namespace App\Console\Commands;

use App\Domain\Blog\Models\Post;
use App\Services\OpenAITranslationService;
use Illuminate\Console\Command;
use App\Services\ArticlePublisherService;
use Illuminate\Support\Facades\Http;

class PublishTranslatedArticles extends Command
{
    protected $signature = 'publish:translated-article {postId}';
    protected $description = 'Translate and publish a specific article with ChatGPT to multiple websites';
    protected $articlePublisher;

    public function __construct(ArticlePublisherService $articlePublisher)
    {
        parent::__construct();
        $this->articlePublisher = $articlePublisher;
    }

    public function handle()
    {
        // Fetch the article ID from the command arguments
        $postId = $this->argument('postId');
        $wordpressSites = json_decode(env('WORDPRESS_SITES'), true);
        if (!$wordpressSites) {
            $this->error('Failed to load WordPress sites configuration.');
            return;
        }
        // Retrieve the article from the database
        $post = Post::find($postId);

        if (!$post) {
            $this->error("Article with ID {$postId} not found.");
            return 1; // Return an error code
        }

        foreach ($wordpressSites as $lang => $site) {
            $translatedArticle = (new OpenAITranslationService())->TranslateArticle($post,$lang);

            if (!$translatedArticle) {
                $this->error("Translation failed for language: $lang");
                continue;
            }

            $this->publishToWordPress(
                $site['url'],
                $site['user'],
                $site['password'],
                $translatedArticle['title'],
                $translatedArticle['content'],
                $lang
            );
            $this->info('Article translated and published successfully! in : '.$site['url']);
        }

        $this->info('All Article translated and published successfully!');
        return 0; // Success code
    }

    public function publishToWordPress($url, $username, $password, $title, $content, $lang)
    {
        $this->info("Starting publication process for language: $lang");
        $this->info("Original Title: $title");

        // Process title
        $title = preg_replace('/\[([^\]]*)\]/', '$1', $title);
        $title = str_replace('website_title', env('Website_name_en'), $title);
        $title = str_replace('website_name', env('Website_name_en'), $title);
        $this->info("Processed Title: $title");

        // Process content
        $this->info("Processing content...");
        $content = preg_replace('/\[([^\]]*)\]/', '$1', $content);
        $content = str_replace('website_name', env('Website_name_en'), $content);
        $content = str_replace('website_title', env('Website_name_en'), $content);

        // URL processing info
        $this->info("Processing URLs for language: $lang");

        // Find and replace URLs in href attributes
        $content = preg_replace_callback(
            '/href=["\']([^"\']+)["\']/',
            function($matches) use ($lang) {
                $oldUrl = $matches[1];

                // Skip if it's an anchor or mailto link
                if (str_starts_with($oldUrl, '#') || str_starts_with($oldUrl, 'mailto:')) {
                    return 'href="' . $oldUrl . '"';
                }

                // Parse the URL
                $parsedUrl = parse_url($oldUrl);

                if (!isset($parsedUrl['host'])) {
                    return 'href="' . $oldUrl . '"';
                }

                // Reconstruct the base URL (scheme + host)
                $newUrl = $parsedUrl['scheme'] . '://' . $parsedUrl['host'];
                $newUrl .= '/' . $lang;

                if (isset($parsedUrl['path'])) {
                    $path = $parsedUrl['path'];
                    $path = preg_replace('/^\/[a-z]{2}\//', '/', $path);
                    $newUrl .= $path;
                }

                if (isset($parsedUrl['query'])) {
                    $newUrl .= '?' . $parsedUrl['query'];
                }

                if (isset($parsedUrl['fragment'])) {
                    $newUrl .= '#' . $parsedUrl['fragment'];
                }

                return 'href="' . $newUrl . '"';
            },
            $content
        );

        try {
            $this->info("Attempting to publish to WordPress...");
            $this->info("Target URL: $url");

            $response = Http::withBasicAuth($username, $password)
                ->post($url, [
                    'title'   => $title,
                    'content' => $content,
                    'status'  => 'publish',
                ]);

            if ($response->successful()) {
                $responseData = $response->json();
                $postId = $responseData['id'] ?? 'N/A';
                $postUrl = $responseData['link'] ?? 'N/A';

                $this->info("✅ Publication successful!");
                $this->info("----------------------------------------");
                $this->info("Publication Details:");
                $this->info("- Language: $lang");
                $this->info("- Post ID: $postId");
                $this->info("- Title: $title");
                $this->info("- URL: $postUrl");
                $this->info("----------------------------------------");
            } else {
                $this->error("❌ Publication failed!");
                $this->error("----------------------------------------");
                $this->error("Error Details:");
                $this->error("- Language: $lang");
                $this->error("- Target URL: $url");
                $this->error("- Response: " . $response->body());
                $this->error("----------------------------------------");
            }
        } catch (\Exception $e) {
            $this->error("❌ Publication error!");
            $this->error("----------------------------------------");
            $this->error("Error Details:");
            $this->error("- Language: $lang");
            $this->error("- Target URL: $url");
            $this->error("- Error Message: " . $e->getMessage());
            $this->error("----------------------------------------");
        }
    }
}
